/*****************************************************************************
 * freetype.c : Put text on the video, using freetype2
 *****************************************************************************
 * Copyright (C) 2002 - 2015 VLC authors and VideoLAN
 *
 * Authors: Sigmund Augdal Helberg <dnumgis@videolan.org>
 *          Gildas Bazin <gbazin@videolan.org>
 *          Bernie Purcell <bitmap@videolan.org>
 *          Jean-Baptiste Kempf <jb@videolan.org>
 *          Felix Paul Kühne <fkuehne@videolan.org>
 *          Salah-Eddin Shaban <salshaaban@gmail.com>
 *
 * This program is free software; you can redistribute it and/or modify it
 * under the terms of the GNU Lesser General Public License as published by
 * the Free Software Foundation; either version 2.1 of the License, or
 * (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. See the
 * GNU Lesser General Public License for more details.
 *
 * You should have received a copy of the GNU Lesser General Public License
 * along with this program; if not, write to the Free Software Foundation, Inc.,
 * Inc., 51 Franklin Street, Fifth Floor, Boston MA 02110-1301, USA.
 *****************************************************************************/

/*****************************************************************************
 * Preamble
 *****************************************************************************/

#ifdef HAVE_CONFIG_H
# include "config.h"
#endif
#include <math.h>

#include <vlc_common.h>
#include <vlc_plugin.h>
#include <vlc_input.h>                             /* vlc_input_attachment_* */
#include <vlc_filter.h>                                      /* filter_sys_t */
#include <vlc_subpicture.h>
#include <vlc_text_style.h>                                   /* text_style_t*/
#include <vlc_charset.h>

#include <assert.h>

#include "platform_fonts.h"
#include "freetype.h"
#include "text_layout.h"
#include "blend/rgb.h"
#include "blend/yuv.h"

/*****************************************************************************
 * Module descriptor
 *****************************************************************************/
static int  Create ( filter_t * );
static void Destroy( filter_t * );

#define FONT_TEXT N_("Font")
#define MONOSPACE_FONT_TEXT N_("Monospace Font")

#define FAMILY_LONGTEXT N_("Font family for the font you want to use")
#define FONT_LONGTEXT N_("Font file for the font you want to use")

#define OPACITY_TEXT N_("Text opacity")
#define OPACITY_LONGTEXT N_("The opacity (inverse of transparency) of the " \
    "text that will be rendered on the video. 0 = transparent, " \
    "255 = totally opaque." )
#define COLOR_TEXT N_("Text default color")
#define COLOR_LONGTEXT N_("The color of the text that will be rendered on "\
    "the video. This must be an hexadecimal (like HTML colors). The first two "\
    "chars are for red, then green, then blue. #000000 = black, #FF0000 = red,"\
    " #00FF00 = green, #FFFF00 = yellow (red + green), #FFFFFF = white" )

#define BOLD_TEXT N_("Force bold")

#define BG_OPACITY_TEXT N_("Background opacity")
#define BG_COLOR_TEXT N_("Background color")

#define OUTLINE_OPACITY_TEXT N_("Outline opacity")
#define OUTLINE_COLOR_TEXT N_("Outline color")
#define OUTLINE_THICKNESS_TEXT N_("Outline thickness")

#define SHADOW_OPACITY_TEXT N_("Shadow opacity")
#define SHADOW_COLOR_TEXT N_("Shadow color")
#define SHADOW_ANGLE_TEXT N_("Shadow angle")
#define SHADOW_DISTANCE_TEXT N_("Shadow distance")
#define CACHE_SIZE_TEXT N_("Cache size")
#define CACHE_SIZE_LONGTEXT N_("Cache size in kBytes")

#define TEXT_DIRECTION_TEXT N_("Text direction")
#define TEXT_DIRECTION_LONGTEXT N_("Paragraph base direction for the Unicode bi-directional algorithm.")


#define YUVP_TEXT N_("Use YUVP renderer")
#define YUVP_LONGTEXT N_("This renders the font using \"paletized YUV\". " \
  "This option is only needed if you want to encode into DVB subtitles" )

static const int pi_color_values[] = {
  0x00000000, 0x00808080, 0x00C0C0C0, 0x00FFFFFF, 0x00800000,
  0x00FF0000, 0x00FF00FF, 0x00FFFF00, 0x00808000, 0x00008000, 0x00008080,
  0x0000FF00, 0x00800080, 0x00000080, 0x000000FF, 0x0000FFFF };

static const char *const ppsz_color_descriptions[] = {
  N_("Black"), N_("Gray"), N_("Silver"), N_("White"), N_("Maroon"),
  N_("Red"), N_("Fuchsia"), N_("Yellow"), N_("Olive"), N_("Green"), N_("Teal"),
  N_("Lime"), N_("Purple"), N_("Navy"), N_("Blue"), N_("Aqua") };

static const int pi_outline_thickness[] = {
    0, 2, 4, 6,
};
static const char *const ppsz_outline_thickness[] = {
    N_("None"), N_("Thin"), N_("Normal"), N_("Thick"),
};

#ifdef HAVE_FRIBIDI
static const int pi_text_direction[] = {
    0, 1, 2,
};
static const char *const ppsz_text_direction[] = {
    N_("Left to right"), N_("Right to left"), N_("Auto"),
};
#endif

vlc_module_begin ()
    set_shortname( N_("Text renderer"))
    set_description( N_("Freetype2 font renderer") )
    set_subcategory( SUBCAT_VIDEO_SUBPIC )

#ifdef HAVE_GET_FONT_BY_FAMILY_NAME
    add_font("freetype-font", DEFAULT_FAMILY, FONT_TEXT, FAMILY_LONGTEXT)
    add_font("freetype-monofont", DEFAULT_MONOSPACE_FAMILY,
             MONOSPACE_FONT_TEXT, FAMILY_LONGTEXT)
#else
    add_loadfile("freetype-font", DEFAULT_FONT_FILE, FONT_TEXT, FONT_LONGTEXT)
    add_loadfile("freetype-monofont", DEFAULT_MONOSPACE_FONT_FILE,
                 MONOSPACE_FONT_TEXT, FONT_LONGTEXT)
#endif

    /* opacity valid on 0..255, with default 255 = fully opaque */
    add_integer_with_range( "freetype-opacity", 255, 0, 255,
        OPACITY_TEXT, OPACITY_LONGTEXT )
        change_safe()

    /* hook to the color values list, with default 0x00ffffff = white */
    add_rgb("freetype-color", 0x00FFFFFF, COLOR_TEXT, COLOR_LONGTEXT)
        change_integer_list( pi_color_values, ppsz_color_descriptions )
        change_safe()

    add_bool( "freetype-bold", false, BOLD_TEXT, NULL )
        change_safe()

    add_integer_with_range( "freetype-background-opacity", 0, 0, 255,
                            BG_OPACITY_TEXT, NULL )
        change_safe()
    add_rgb("freetype-background-color", 0x00000000, BG_COLOR_TEXT, NULL)
        change_integer_list( pi_color_values, ppsz_color_descriptions )
        change_safe()

    add_integer_with_range( "freetype-outline-opacity", 255, 0, 255,
                            OUTLINE_OPACITY_TEXT, NULL )
        change_safe()
    add_rgb("freetype-outline-color", 0x00000000, OUTLINE_COLOR_TEXT, NULL)
        change_integer_list( pi_color_values, ppsz_color_descriptions )
        change_safe()
    add_integer_with_range( "freetype-outline-thickness", 4, 0, 50, OUTLINE_THICKNESS_TEXT,
             NULL )
        change_integer_list( pi_outline_thickness, ppsz_outline_thickness )
        change_safe()

    add_integer_with_range( "freetype-shadow-opacity", 128, 0, 255,
                            SHADOW_OPACITY_TEXT, NULL )
        change_safe()
    add_rgb("freetype-shadow-color", 0x00000000, SHADOW_COLOR_TEXT, NULL)
        change_integer_list( pi_color_values, ppsz_color_descriptions )
        change_safe()
    add_float_with_range( "freetype-shadow-angle", -45, -360, 360,
                          SHADOW_ANGLE_TEXT, NULL )
        change_safe()
    add_float_with_range( "freetype-shadow-distance", 0.06, 0.0, 1.0,
                          SHADOW_DISTANCE_TEXT, NULL )
        change_safe()

    add_integer_with_range( "freetype-cache-size", 200, 25, (UINT32_MAX >> 10),
                            CACHE_SIZE_TEXT, CACHE_SIZE_LONGTEXT )
        change_safe()

    add_obsolete_integer( "freetype-fontsize" ) /* since 4.0.0 */
    add_obsolete_integer( "freetype-rel-fontsize" ) /* since 4.0.0 */

    add_bool( "freetype-yuvp", false, YUVP_TEXT,
              YUVP_LONGTEXT )

#ifdef HAVE_FRIBIDI
    add_integer_with_range( "freetype-text-direction", 0, 0, 2, TEXT_DIRECTION_TEXT,
                            TEXT_DIRECTION_LONGTEXT )
        change_integer_list( pi_text_direction, ppsz_text_direction )
        change_safe()
#endif

    add_shortcut( "text" )
    set_callback_text_renderer( Create, 100 )
vlc_module_end ()

/* */

static FT_Vector GetAlignedOffset( const line_desc_t *p_line,
                                   const FT_BBox *p_textbbox,
                                   int i_align )
{
    FT_Vector offsets = { 0, 0 };

    /* aligns left to textbbox's min first */
    offsets.x = p_textbbox->xMin - p_line->bbox.xMin;

    const int i_text_width = p_textbbox->xMax - p_textbbox->xMin;
    if ( p_line->i_width < i_text_width &&
        (i_align & SUBPICTURE_ALIGN_LEFT) == 0 )
    {
        /* Left offset to take into account alignment */
        if( i_align & SUBPICTURE_ALIGN_RIGHT )
            offsets.x += ( i_text_width - p_line->i_width );
        else /* center */
            offsets.x += ( i_text_width - p_line->i_width ) / 2;
    }
    /* else already left aligned */

    return offsets;
}

static bool IsSupportedAttachment( const char *psz_mime )
{
    static const char * fontMimeTypes[] =
    {
        "application/x-truetype-font", // TTF
        "application/x-font-otf",  // OTF
        "application/font-sfnt",
        "font/ttf",
        "font/otf",
        "font/sfnt",
    };

    for( size_t i=0; i<ARRAY_SIZE(fontMimeTypes); ++i )
    {
        if( !strcmp( psz_mime, fontMimeTypes[i] ) )
            return true;
    }

    return false;
}

/*****************************************************************************
 * Make any TTF/OTF fonts present in the attachments of the media file
 * and store them for later use by the FreeType Engine
 *****************************************************************************/
static int LoadFontsFromAttachments( filter_t *p_filter )
{
    filter_sys_t         *p_sys = p_filter->p_sys;
    input_attachment_t  **pp_attachments;
    int                   i_attachments_cnt;
    FT_Face               p_face = NULL;

    if( filter_GetInputAttachments( p_filter, &pp_attachments, &i_attachments_cnt ) )
        return VLC_EGENERIC;

    p_sys->i_font_attachments = 0;
    p_sys->pp_font_attachments = vlc_alloc( i_attachments_cnt, sizeof(*p_sys->pp_font_attachments));
    if( !p_sys->pp_font_attachments )
    {
        for( int i = 0; i < i_attachments_cnt; ++i )
            vlc_input_attachment_Release( pp_attachments[ i ] );
        free( pp_attachments );
        return VLC_ENOMEM;
    }

    int k = 0;
    for( ; k < i_attachments_cnt; k++ )
    {
        input_attachment_t *p_attach = pp_attachments[k];

        if( p_attach->i_data > 0 && p_attach->p_data &&
            IsSupportedAttachment( p_attach->psz_mime ) )
        {
            p_sys->pp_font_attachments[ p_sys->i_font_attachments++ ] = p_attach;

            int i_font_idx = 0;

            while( 0 == FT_New_Memory_Face( p_sys->p_library,
                                            p_attach->p_data,
                                            p_attach->i_data,
                                            i_font_idx,
                                            &p_face ))
            {
                int i_flags = 0;
                if( p_face->style_flags & FT_STYLE_FLAG_BOLD )
                    i_flags |= VLC_FONT_FLAG_BOLD;
                if( p_face->style_flags & FT_STYLE_FLAG_ITALIC )
                    i_flags |= VLC_FONT_FLAG_ITALIC;

                vlc_family_t *p_family = DeclareNewFamily( p_sys->fs, p_face->family_name );
                if( unlikely( !p_family ) )
                    goto error;

                char *psz_fontfile;
                if( asprintf( &psz_fontfile, ":/%d",
                              p_sys->i_font_attachments - 1 ) < 0
                 || !NewFont( psz_fontfile, i_font_idx, i_flags, p_family ) )
                    goto error;

                FT_Done_Face( p_face );
                p_face = NULL;

                /* Add font attachment to the "attachments" fallback list */
                DeclareFamilyAsAttachMenFallback( p_sys->fs, p_family );

                i_font_idx++;
            }
        }
        else
        {
            vlc_input_attachment_Release( p_attach );
        }
    }

    free( pp_attachments );

    return VLC_SUCCESS;

error:
    if( p_face )
        FT_Done_Face( p_face );

    for( int i = k + 1; i < i_attachments_cnt; ++i )
        vlc_input_attachment_Release( pp_attachments[ i ] );

    free( pp_attachments );
    return VLC_ENOMEM;
}

/*****************************************************************************
 * RenderYUVP: place string in picture
 *****************************************************************************
 * This function merges the previously rendered freetype glyphs into a picture
 *****************************************************************************/
static void RenderYUVP( const subpicture_region_t *p_region_in,
                       subpicture_region_t *p_region,
                       const line_desc_t *p_line,
                       const FT_BBox *p_regionbbox,
                       const FT_BBox *p_bbox )
{
    static const uint8_t pi_gamma[16] =
        {0x00, 0x52, 0x84, 0x96, 0xb8, 0xca, 0xdc, 0xee, 0xff,
          0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff};

    uint8_t *p_dst;
    int i, i_pitch;
    unsigned int x, y;
    uint8_t i_y, i_u, i_v; /* YUV values, derived from incoming RGB */

    /* Calculate text color components
     * Only use the first color */
    const int i_alpha = p_line->p_character[0].p_style->i_font_alpha;
    YUVFromXRGB( p_line->p_character[0].p_style->i_font_color, &i_y, &i_u, &i_v );

    /* Build palette */
    p_region->p_picture->format.p_palette->i_entries = 16;
    for( i = 0; i < 8; i++ )
    {
        p_region->p_picture->format.p_palette->palette[i][0] = 0;
        p_region->p_picture->format.p_palette->palette[i][1] = 0x80;
        p_region->p_picture->format.p_palette->palette[i][2] = 0x80;
        p_region->p_picture->format.p_palette->palette[i][3] = (int)pi_gamma[i] * i_alpha / 255;
    }
    for( i = 8; i < p_region->p_picture->format.p_palette->i_entries; i++ )
    {
        p_region->p_picture->format.p_palette->palette[i][0] = i * 16 * i_y / 256;
        p_region->p_picture->format.p_palette->palette[i][1] = i_u;
        p_region->p_picture->format.p_palette->palette[i][2] = i_v;
        p_region->p_picture->format.p_palette->palette[i][3] = (int)pi_gamma[i] * i_alpha / 255;
    }

    p_dst = p_region->p_picture->Y_PIXELS;
    i_pitch = p_region->p_picture->Y_PITCH;

    /* Initialize the region pixels */
    memset( p_dst, 0, i_pitch * p_region->p_picture->format.i_height );

    for( ; p_line != NULL; p_line = p_line->p_next )
    {
        FT_Vector offset = GetAlignedOffset( p_line, p_bbox, p_region_in->i_align );

        for( i = 0; i < p_line->i_character_count; i++ )
        {
            const line_character_t *ch = &p_line->p_character[i];
            FT_BitmapGlyph p_glyph = ch->p_glyph;

            int i_glyph_y = offset.y + p_regionbbox->yMax - p_glyph->top + p_line->origin.y;
            int i_glyph_x = offset.x + p_glyph->left - p_regionbbox->xMin;

            for( y = 0; y < p_glyph->bitmap.rows; y++ )
            {
                for( x = 0; x < p_glyph->bitmap.width; x++ )
                {
                    if( p_glyph->bitmap.buffer[y * p_glyph->bitmap.pitch + x] )
                        p_dst[(i_glyph_y + y) * i_pitch + (i_glyph_x + x)] =
                            (p_glyph->bitmap.buffer[y * p_glyph->bitmap.pitch + x] + 8)/16;
                }
            }
        }
    }

    /* Outlining (find something better than nearest neighbour filtering ?) */
    if( 1 )
    {
        p_dst = p_region->p_picture->Y_PIXELS;
        uint8_t *p_top = p_dst; /* Use 1st line as a cache */
        uint8_t left, current;

        for( y = 1; y < p_region->p_picture->format.i_height - 1; y++ )
        {
            if( y > 1 ) memcpy( p_top, p_dst, p_region->p_picture->format.i_width );
            p_dst += p_region->p_picture->Y_PITCH;
            left = 0;

            for( x = 1; x < p_region->p_picture->format.i_width - 1; x++ )
            {
                current = p_dst[x];
                p_dst[x] = ( 8 * (int)p_dst[x] + left + p_dst[x+1] + p_top[x -1]+ p_top[x] + p_top[x+1] +
                             p_dst[x -1 + p_region->p_picture->Y_PITCH ] + p_dst[x + p_region->p_picture->Y_PITCH] + p_dst[x + 1 + p_region->p_picture->Y_PITCH]) / 16;
                left = current;
            }
        }
        memset( p_top, 0, p_region->p_picture->format.i_width );
    }
}

/*****************************************************************************
 * RenderYUVA: place string in picture
 *****************************************************************************
 * This function merges the previously rendered freetype glyphs into a picture
 *****************************************************************************/

static void RenderBackground( const subpicture_region_t *p_region_in,
                                     const line_desc_t *p_line_head,
                                     const FT_BBox *p_regionbbox,
                                     const FT_BBox *p_paddedbbox,
                                     const FT_BBox *p_textbbox,
                                     picture_t *p_picture,
                                     const ft_drawing_functions *draw )
{
    for( const line_desc_t *p_line = p_line_head; p_line != NULL; p_line = p_line->p_next )
    {
        FT_Vector offset = GetAlignedOffset( p_line, p_textbbox, p_region_in->text_flags & SUBPICTURE_ALIGN_MASK );

        FT_BBox linebgbox = p_line->bbox;
        linebgbox.xMin += offset.x;
        linebgbox.xMax += offset.x;
        linebgbox.yMax += offset.y;
        linebgbox.yMin += offset.y;

        if( p_line->i_first_visible_char_index < 0 )
            continue; /* only spaces */

        /* add padding */
        linebgbox.yMax += (p_paddedbbox->yMax - p_textbbox->yMax);
        linebgbox.yMin -= (p_textbbox->yMin - p_paddedbbox->yMin);
        linebgbox.xMin -= (p_textbbox->xMin - p_paddedbbox->xMin);
        linebgbox.xMax += (p_paddedbbox->xMax - p_textbbox->xMax);

        /* Compute lower boundary for the background
           continue down to next line top */
        if( p_line->p_next && p_line->p_next->i_first_visible_char_index >= 0 )
            linebgbox.yMin = __MIN(linebgbox.yMin, p_line->bbox.yMin - (p_line->bbox.yMin - p_line->p_next->bbox.yMax));

        /* Setup color for the background */
        const text_style_t *p_prev_style = p_line->p_character[p_line->i_first_visible_char_index].p_style;

        FT_BBox segmentbgbox = linebgbox;
        int i_char_index = p_line->i_first_visible_char_index;
        /* Compute the background for the line (identify leading/trailing space) */
        if( i_char_index > 0 )
        {
            segmentbgbox.xMin = offset.x +
                                p_line->p_character[p_line->i_first_visible_char_index].bbox.xMin -
                                /* padding offset */ (p_textbbox->xMin - p_paddedbbox->xMin);
        }

        while( i_char_index <= p_line->i_last_visible_char_index )
        {
            /* find last char having the same style */
            int i_seg_end = i_char_index;
            while( i_seg_end + 1 <= p_line->i_last_visible_char_index &&
                   p_prev_style == p_line->p_character[i_seg_end + 1].p_style )
            {
                i_seg_end++;
            }

            /* Find right boundary for bounding box for background */
            segmentbgbox.xMax = offset.x + p_line->p_character[i_seg_end].bbox.xMax;
            if( i_seg_end == p_line->i_last_visible_char_index ) /* add padding on last */
                segmentbgbox.xMax += (p_paddedbbox->xMax - p_textbbox->xMax);

            const line_character_t *p_char = &p_line->p_character[i_char_index];
            if( p_char->p_style->i_style_flags & STYLE_BACKGROUND )
            {
                uint8_t i_x, i_y, i_z;
                draw->extract( p_char->p_style->i_background_color,
                                 &i_x, &i_y, &i_z );
                const uint8_t i_alpha = p_char->p_style->i_background_alpha;

                /* Render the actual background */
                if( i_alpha != STYLE_ALPHA_TRANSPARENT )
                {
                    /* rebase and clip to SCREEN coordinates */
                    FT_BBox absbox =
                    {
                        .xMin = __MAX(0, segmentbgbox.xMin - p_regionbbox->xMin),
                        .xMax = VLC_CLIP(segmentbgbox.xMax - p_regionbbox->xMin,
                                         0, p_picture->format.i_visible_width),
                        .yMin = VLC_CLIP(p_regionbbox->yMax - segmentbgbox.yMin,
                                         0, p_picture->format.i_visible_height),
                        .yMax = __MAX(0, p_regionbbox->yMax - segmentbgbox.yMax),
                    };

                    draw->fill( p_picture, i_alpha, i_x, i_y, i_z,
                               absbox.xMin, absbox.yMax,
                               absbox.xMax - absbox.xMin, absbox.yMin - absbox.yMax );
                }
            }

            segmentbgbox.xMin = segmentbgbox.xMax;
            i_char_index = i_seg_end + 1;
            p_prev_style = p_line->p_character->p_style;
        }

    }
}

static void RenderCharAXYZ( filter_t *p_filter,
                           picture_t *p_picture,
                           const line_desc_t *p_line,
                           int i_offset_x,
                           int i_offset_y,
                           int g,
                           const ft_drawing_functions *draw )
{
    VLC_UNUSED(p_filter);
    /* Render all glyphs and underline/strikethrough */
    for( int i = p_line->i_first_visible_char_index; i <= p_line->i_last_visible_char_index; i++ )
    {
        const line_character_t *ch = &p_line->p_character[i];
        const FT_BitmapGlyph p_glyph = g == 0 ? ch->p_shadow : g == 1 ? ch->p_outline : ch->p_glyph;
        if( !p_glyph )
            continue;

        uint8_t i_a = ch->p_style->i_font_alpha;

        uint32_t i_color;
        switch (g) {/* Apply font alpha ratio to shadow/outline alpha */
        case 0:
            i_a     = i_a * ch->p_style->i_shadow_alpha / 255;
            i_color = ch->p_style->i_shadow_color;
            break;
        case 1:
            i_a     = i_a * ch->p_style->i_outline_alpha / 255;
            i_color = ch->p_style->i_outline_color;
            break;
        default:
            i_color = ch->p_style->i_font_color;
            break;
        }

        const line_desc_t *p_rubydesc = ch->p_ruby ? ch->p_ruby->p_laid : NULL;
        if(p_rubydesc)
        {
            RenderCharAXYZ( p_filter,
                            p_picture,
                            p_rubydesc,
                            i_offset_x + p_rubydesc->origin.x,
                            i_offset_y - p_rubydesc->origin.y,
                            2,
                            draw );
        }

        /* Don't render if invisible or not wanted */
        if( i_a == STYLE_ALPHA_TRANSPARENT ||
           (g == 0 && 0 == (ch->p_style->i_style_flags & STYLE_SHADOW) ) ||
           (g == 1 && 0 == (ch->p_style->i_style_flags & STYLE_OUTLINE) )
          )
            continue;

        uint8_t i_x, i_y, i_z;
        draw->extract( i_color, &i_x, &i_y, &i_z );

        int i_glyph_y = i_offset_y - p_glyph->top;
        int i_glyph_x = i_offset_x + p_glyph->left;

        draw->blend( p_picture, i_glyph_x, i_glyph_y,
                    i_a, i_x, i_y, i_z, p_glyph );

        /* underline/strikethrough are only rendered for the normal glyph */
        if( g == 2 && ch->i_line_thickness > 0 )
            BlendAXYZLine( p_picture,
                           i_glyph_x, i_glyph_y + ch->bbox.yMax,
                           i_a, i_x, i_y, i_z,
                           &ch[0],
                           i + 1 < p_line->i_character_count ? &ch[1] : NULL,
                           draw );
    }
}

static inline void RenderAXYZ( filter_t *p_filter,
                              const subpicture_region_t *p_region_in,
                              subpicture_region_t *p_region,
                              const line_desc_t *p_line_head,
                              const FT_BBox *p_regionbbox,
                              const FT_BBox *p_paddedtextbbox,
                              const FT_BBox *p_textbbox,
                              const ft_drawing_functions *draw )
{
    filter_sys_t *p_sys = p_filter->p_sys;

    picture_t *p_picture = p_region->p_picture;

    /* Initialize the picture background */
    const text_style_t *p_style = p_sys->p_default_style;
    uint8_t i_x, i_y, i_z;

    if (p_region_in->text_flags & VLC_SUBPIC_TEXT_FLAG_NO_REGION_BG) {
        /* Render the background just under the text */
        draw->fill( p_picture, STYLE_ALPHA_TRANSPARENT, 0x00, 0x00, 0x00,
                   0, 0, p_region->fmt.i_visible_width, p_region->fmt.i_visible_height );
    } else {
        /* Render background under entire subpicture block */
        draw->extract( p_style->i_background_color, &i_x, &i_y, &i_z );
        draw->fill( p_picture, p_style->i_background_alpha, i_x, i_y, i_z,
                   0, 0, p_region->fmt.i_visible_width, p_region->fmt.i_visible_height );
    }

    /* Render text's background (from decoder) if any */
    RenderBackground(p_region_in, p_line_head,
                     p_regionbbox, p_paddedtextbbox, p_textbbox,
                     p_picture, draw);

    /* Render shadow then outline and then normal glyphs */
    for( int g = 0; g < 3; g++ )
    {
        /* Render all lines */
        for( const line_desc_t *p_line = p_line_head; p_line != NULL; p_line = p_line->p_next )
        {
            FT_Vector offset = GetAlignedOffset( p_line, p_textbbox, p_region_in->text_flags & SUBPICTURE_ALIGN_MASK );

            int i_glyph_offset_y = offset.y + p_regionbbox->yMax + p_line->origin.y;
            int i_glyph_offset_x = offset.x - p_regionbbox->xMin;

            RenderCharAXYZ( p_filter, p_picture, p_line,
                            i_glyph_offset_x, i_glyph_offset_y, g,
                            draw );
        }
    }
}

static void UpdateDefaultLiveStyles( filter_t *p_filter )
{
    filter_sys_t *p_sys = p_filter->p_sys;
    text_style_t *p_style = p_sys->p_default_style;

    p_style->i_font_color = var_InheritInteger( p_filter, "freetype-color" );

    p_style->i_background_alpha = var_InheritInteger( p_filter, "freetype-background-opacity" );
    p_style->i_background_color = var_InheritInteger( p_filter, "freetype-background-color" );

    p_sys->i_outline_thickness = var_InheritInteger( p_filter, "freetype-outline-thickness" );
}

static void FillDefaultStyles( filter_t *p_filter )
{
    filter_sys_t *p_sys = p_filter->p_sys;

#ifdef HAVE_GET_FONT_BY_FAMILY_NAME
    p_sys->p_default_style->psz_fontname = var_InheritString( p_filter, "freetype-font" );
#endif
    /* Set default psz_fontname */
    if( !p_sys->p_default_style->psz_fontname || !*p_sys->p_default_style->psz_fontname )
    {
        free( p_sys->p_default_style->psz_fontname );
        p_sys->p_default_style->psz_fontname = strdup( DEFAULT_FAMILY );
    }

#ifdef HAVE_GET_FONT_BY_FAMILY_NAME
    p_sys->p_default_style->psz_monofontname = var_InheritString( p_filter, "freetype-monofont" );
#endif
    /* set default psz_monofontname */
    if( !p_sys->p_default_style->psz_monofontname || !*p_sys->p_default_style->psz_monofontname )
    {
        free( p_sys->p_default_style->psz_monofontname );
        p_sys->p_default_style->psz_monofontname = strdup( DEFAULT_MONOSPACE_FAMILY );
    }

    UpdateDefaultLiveStyles( p_filter );

    p_sys->p_default_style->i_font_alpha = var_InheritInteger( p_filter, "freetype-opacity" );

    p_sys->p_default_style->i_outline_alpha = var_InheritInteger( p_filter, "freetype-outline-opacity" );
    p_sys->p_default_style->i_outline_color = var_InheritInteger( p_filter, "freetype-outline-color" );

    p_sys->p_default_style->i_shadow_alpha = var_InheritInteger( p_filter, "freetype-shadow-opacity" );
    p_sys->p_default_style->i_shadow_color = var_InheritInteger( p_filter, "freetype-shadow-color" );

    p_sys->p_default_style->i_font_size = 0;
    p_sys->p_default_style->i_style_flags |= STYLE_SHADOW;
    p_sys->p_default_style->i_features |= STYLE_HAS_FLAGS;

    if( var_InheritBool( p_filter, "freetype-bold" ) )
    {
        p_sys->p_forced_style->i_style_flags |= STYLE_BOLD;
        p_sys->p_forced_style->i_features |= STYLE_HAS_FLAGS;
    }

    /* Apply forced styles to defaults, if any */
    text_style_Merge( p_sys->p_default_style, p_sys->p_forced_style, true );
}

static void FreeRubyBlockArray( ruby_block_t **pp_array, size_t i_array )
{
    ruby_block_t *p_lyt = NULL;
    for( size_t i = 0; i< i_array; i++ )
    {
        if( p_lyt != pp_array[i] )
        {
            p_lyt = pp_array[i];
            if( p_lyt )
            {
                free( p_lyt->p_uchars );
                text_style_Delete( p_lyt->p_style );
                if( p_lyt->p_laid )
                    FreeLines( p_lyt->p_laid );
                free( p_lyt );
            }
        }
    }
    free( pp_array );
}

static void FreeStylesArray( text_style_t **pp_styles, size_t i_styles )
{
    text_style_t *p_style = NULL;
    for( size_t i = 0; i< i_styles; i++ )
    {
        if( p_style != pp_styles[i] )
        {
            p_style = pp_styles[i];
            text_style_Delete( p_style );
        }
    }
    free( pp_styles );
}

#ifdef __OS2__
static void *ToUCS4( const char *in, size_t *outsize )
{
    uint16_t *psz_ucs2;
    uint32_t *psz_ucs4;
    size_t i_bytes;

    psz_ucs2 = ToCharset("UCS-2LE", in, &i_bytes );
    if( unlikely( !psz_ucs2 ) )
        return NULL;

    i_bytes <<= 1;
    /* Realloc including NULL-terminator */
    psz_ucs4 = realloc( psz_ucs2, i_bytes + sizeof( *psz_ucs4 ));
    if( unlikely( !psz_ucs4 ) )
    {
        free( psz_ucs2 );

        return NULL;
    }

    psz_ucs2 = ( uint16_t * )psz_ucs4;
    /* Copy including NULL-terminator */
    for( int i = i_bytes / sizeof( *psz_ucs4 ); i >= 0; --i )
        psz_ucs4[ i ] = psz_ucs2[ i ];

    *outsize = i_bytes;

    return psz_ucs4;
}
#else
# define ToUCS4( in, outsize ) ToCharset( FREETYPE_TO_UCS, ( in ), ( outsize ))
#endif

static size_t AddTextAndStyles( filter_sys_t *p_sys,
                                const char *psz_text, const char *psz_rt,
                                const text_style_t *p_style,
                                layout_text_block_t *p_text_block )
{
    text_style_t *p_mgstyle = NULL;

    /* Convert chars to unicode */
    size_t i_bytes;
    uni_char_t *p_ucs4 = ToUCS4( psz_text, &i_bytes );
    if( !p_ucs4 )
        return 0;

    const size_t i_newchars = i_bytes / 4;
    const size_t i_new_count = p_text_block->i_count + i_newchars;
    if( SIZE_MAX / 4 < i_new_count )
        goto error;

    size_t i_realloc = i_new_count * 4;
    void *p_realloc = realloc( p_text_block->p_uchars, i_realloc );
    if( unlikely(!p_realloc) )
        goto error;
    p_text_block->p_uchars = p_realloc;

    /* We want one per segment shared text_style_t* per unicode character */
    if( SIZE_MAX / sizeof(text_style_t *) < i_new_count )
        goto error;
    i_realloc = i_new_count * sizeof(text_style_t *);
    p_realloc = realloc( p_text_block->pp_styles, i_realloc );
    if ( unlikely(!p_realloc) )
        goto error;
    p_text_block->pp_styles = p_realloc;

    /* Same for ruby text */
    if( SIZE_MAX / sizeof(text_segment_ruby_t *) < i_new_count )
        goto error;
    i_realloc = i_new_count * sizeof(text_segment_ruby_t *);
    p_realloc = realloc( p_text_block->pp_ruby, i_realloc );
    if ( unlikely(!p_realloc) )
        goto error;
    p_text_block->pp_ruby = p_realloc;

    /* Copy data */
    memcpy( &p_text_block->p_uchars[p_text_block->i_count], p_ucs4, i_newchars * 4 );
    free( p_ucs4 );

    p_mgstyle = text_style_Duplicate( p_sys->p_default_style );
    if ( p_mgstyle == NULL )
        return 0;

    if( p_style )
        /* Replace defaults with segment values */
        text_style_Merge( p_mgstyle, p_style, true );

    /* Overwrite any default or value with forced ones */
    text_style_Merge( p_mgstyle, p_sys->p_forced_style, true );

    /* map it to each char of that segment */
    for ( size_t i = 0; i < i_newchars; ++i )
        p_text_block->pp_styles[p_text_block->i_count + i] = p_mgstyle;

    ruby_block_t *p_rubyblock = NULL;
    if( psz_rt )
    {
        p_ucs4 = ToUCS4( psz_rt, &i_bytes );
        if( !p_ucs4 )
            goto error;
        p_rubyblock = malloc(sizeof(ruby_block_t));
        if( p_rubyblock )
        {
            p_rubyblock->p_style = text_style_Duplicate( p_mgstyle );
            if( !p_rubyblock->p_style )
            {
                free( p_rubyblock );
                goto error;
            }
            p_rubyblock->p_style->i_font_size *= 0.4;
            p_rubyblock->p_style->f_font_relsize *= 0.4;
            p_rubyblock->p_uchars = p_ucs4;
            p_rubyblock->i_count = i_bytes / 4;
            p_rubyblock->p_laid = NULL;
        }
        else free( p_ucs4 );
    }
    for ( size_t i = 0; i < i_newchars; ++i )
        p_text_block->pp_ruby[p_text_block->i_count + i] = p_rubyblock;

    /* now safe to update total nb */
    p_text_block->i_count += i_newchars;

    return i_newchars;
error:
    free( p_ucs4 );
    text_style_Delete( p_mgstyle );
    return 0;
}

static size_t SegmentsToTextAndStyles( filter_t *p_filter, const text_segment_t *p_segment,
                                       layout_text_block_t *p_text_block )
{
    size_t i_nb_char = 0;

    for( const text_segment_t *s = p_segment; s != NULL; s = s->p_next )
    {
        if( !s->psz_text || !s->psz_text[0] )
            continue;

        if( s->p_ruby )
        {
            for( const text_segment_ruby_t *p_ruby = s->p_ruby;
                                            p_ruby; p_ruby = p_ruby->p_next )
            {
                i_nb_char += AddTextAndStyles( p_filter->p_sys,
                                               p_ruby->psz_base, p_ruby->psz_rt,
                                               s->style, p_text_block );
            }
        }
        else
        {
            i_nb_char += AddTextAndStyles( p_filter->p_sys,
                                           s->psz_text, NULL,
                                           s->style, p_text_block );
        }
    }

    return i_nb_char;
}

/**
 * This function renders a text subpicture region into another one.
 * It also calculates the size needed for this string, and renders the
 * needed glyphs into memory. It is used as pf_add_string callback in
 * the vout method by this module
 */
static subpicture_region_t *Render( filter_t *p_filter,
                         const subpicture_region_t *p_region_in,
                         const vlc_fourcc_t *p_chroma_list )
{
    filter_sys_t *p_sys = p_filter->p_sys;
    subpicture_region_t *region = NULL;
    bool b_grid = (p_region_in->text_flags & VLC_SUBPIC_TEXT_FLAG_GRID_MODE) != 0;
    p_sys->i_scale = ( b_grid ) ? 100 : var_InheritInteger( p_filter, "sub-text-scale");

    UpdateDefaultLiveStyles( p_filter );

    int i_font_default_size = ConvertToLiveSize( p_filter, p_sys->p_default_style );
    if( !p_sys->p_faceid || i_font_default_size != p_sys->i_font_default_size )
    {
        /* Update the default face to reflect changes in video size or text scaling */
        p_sys->p_faceid = SelectAndLoadFace( p_filter, p_sys->p_default_style, ' ' );
        if( !p_sys->p_faceid )
        {
            msg_Err( p_filter, "Render(): Error loading default face" );
            return NULL;
        }
        p_sys->i_font_default_size = i_font_default_size;
    }

    layout_text_block_t text_block = { 0 };
    text_block.b_balanced = (p_region_in->text_flags & VLC_SUBPIC_TEXT_FLAG_TEXT_NOT_BALANCED) == 0;
    text_block.b_grid = b_grid;
    text_block.i_count = SegmentsToTextAndStyles( p_filter, p_region_in->p_text,
                                                  &text_block );
    if( text_block.i_count == 0 )
    {
        free( text_block.pp_styles );
        free( text_block.p_uchars );
        return NULL;
    }

    /* */
    int rv;
    FT_BBox bbox;
    int i_max_face_height;

    unsigned i_max_width = p_filter->fmt_out.video.i_visible_width;
    if( p_region_in->i_max_width > 0 && (unsigned) p_region_in->i_max_width < i_max_width )
        i_max_width = p_region_in->i_max_width;
    else if( p_region_in->i_x > 0 && (unsigned)p_region_in->i_x < i_max_width )
        i_max_width -= p_region_in->i_x;

    unsigned i_max_height = p_filter->fmt_out.video.i_visible_height;
    if( p_region_in->i_max_height > 0 && (unsigned) p_region_in->i_max_height < i_max_height )
        i_max_height = p_region_in->i_max_height;
    else if( p_region_in->i_y > 0 && (unsigned)p_region_in->i_y < i_max_height )
        i_max_height -= p_region_in->i_y;

    text_block.i_max_width = i_max_width;
    text_block.i_max_height = i_max_height;
    rv = LayoutTextBlock( p_filter, &text_block, &text_block.p_laid, &bbox, &i_max_face_height );

    /* Don't attempt to render text that couldn't be laid out
     * properly. */
    if (!( rv == VLC_SUCCESS && text_block.i_count > 0 && bbox.xMin < bbox.xMax && bbox.yMin < bbox.yMax ))
    {
        rv = VLC_EGENERIC;
        goto done;
    }

    const vlc_fourcc_t p_chroma_list_yuvp[] = { VLC_CODEC_YUVP, 0 };
    const vlc_fourcc_t p_chroma_list_rgba[] = { VLC_CODEC_RGBA, 0 };

    int i_margin = (p_sys->p_default_style->i_background_alpha > 0 && !b_grid)
                    ? i_max_face_height / 4 : 0;

    if( (unsigned)i_margin * 2 >= i_max_width || (unsigned)i_margin * 2 >= i_max_height )
        i_margin = 0;

    if( p_sys->i_forced_chroma == VLC_CODEC_YUVP )
        p_chroma_list = p_chroma_list_yuvp;
    else if( !p_chroma_list || *p_chroma_list == 0 )
        p_chroma_list = p_chroma_list_rgba;

    FT_BBox paddedbbox = bbox;
    paddedbbox.xMin -= i_margin;
    paddedbbox.xMax += i_margin;
    paddedbbox.yMin -= i_margin;
    paddedbbox.yMax += i_margin;

    FT_BBox regionbbox = paddedbbox;

    /* _______regionbbox_______________
     * |                               |
     * |                               |
     * |                               |
     * |     _bbox(<paddedbbox)___     |
     * |    |         rightaligned|    |
     * |    |            textlines|    |
     * |    |_____________________|    |
     * |_______________________________|
     *
     * we need at least 3 bounding boxes.
     * regionbbox containing the whole, including region background pixels
     * paddedbox an enlarged text box when for drawing text background
     * bbox the lines bounding box for all glyphs
     * For simple unstyled subs, bbox == paddedbox == regionbbox
     */

    unsigned outertext_w = (regionbbox.xMax - regionbbox.xMin);
    if( outertext_w < (unsigned) p_region_in->i_max_width )
    {
        if( p_region_in->text_flags & SUBPICTURE_ALIGN_RIGHT )
            regionbbox.xMin -= (p_region_in->i_max_width - outertext_w);
        else if( p_region_in->text_flags & SUBPICTURE_ALIGN_LEFT )
            regionbbox.xMax += (p_region_in->i_max_width - outertext_w);
        else
        {
            regionbbox.xMin -= (p_region_in->i_max_width - outertext_w) / 2;
            regionbbox.xMax += (p_region_in->i_max_width - outertext_w + 1) / 2;
        }
    }

    unsigned outertext_h = (regionbbox.yMax - regionbbox.yMin);
    if( outertext_h < (unsigned) p_region_in->i_max_height )
    {
        if( p_region_in->text_flags & SUBPICTURE_ALIGN_TOP )
            regionbbox.yMin -= (p_region_in->i_max_height - outertext_h);
        else if( p_region_in->text_flags & SUBPICTURE_ALIGN_BOTTOM )
            regionbbox.yMax += (p_region_in->i_max_height - outertext_h);
        else
        {
            regionbbox.yMin -= (p_region_in->i_max_height - outertext_h + 1) / 2;
            regionbbox.yMax += (p_region_in->i_max_height - outertext_h) / 2;
        }
    }

//        unsigned bboxcolor = 0xFF000000;
    /* TODO 4.0. No region self BG color for VLC 3.0 API*/

    /* Avoid useless pixels:
        *        reshrink/trim Region Box to padded text one,
        *        but update offsets to keep position and have same rendering */
    FT_BBox renderbbox;
    // if( (bboxcolor & 0xFF) == 0 )
        renderbbox = paddedbbox;
    // else
    //     renderbbox = regionbbox;

    video_format_t fmt;
    video_format_Init( &fmt, 0 );
    fmt.i_width          =
    fmt.i_visible_width  = renderbbox.xMax - renderbbox.xMin;
    fmt.i_height         =
    fmt.i_visible_height = renderbbox.yMax - renderbbox.yMin;
    fmt.i_sar_num = fmt.i_sar_den = 1;

    for( const vlc_fourcc_t *p_chroma = p_chroma_list; *p_chroma != 0; p_chroma++ )
    {
        /* Create a new subpicture region */
        fmt.i_chroma = *p_chroma;
        region = subpicture_region_New(&fmt);
        if (unlikely(region == NULL))
            continue;

        region->p_picture->format.transfer  = p_region_in->fmt.transfer;
        region->p_picture->format.primaries = p_region_in->fmt.primaries;
        region->p_picture->format.space     = p_region_in->fmt.space;
        region->p_picture->format.mastering = p_region_in->fmt.mastering;

        region->fmt.i_sar_num = p_region_in->fmt.i_sar_num;
        region->fmt.i_sar_den = p_region_in->fmt.i_sar_den;

        if( *p_chroma == VLC_CODEC_YUVP )
            RenderYUVP( p_region_in, region, text_block.p_laid,
                                &renderbbox, &bbox );
        else
        {
            const ft_drawing_functions *func;
            if( *p_chroma == VLC_CODEC_YUVA )
            {
                static const ft_drawing_functions DRAW_YUVA =
                    { .extract = YUVFromXRGB,
                      .fill =    FillYUVAPicture,
                      .blend =   BlendGlyphToYUVA };
                func = &DRAW_YUVA;
            }
            else if( *p_chroma == VLC_CODEC_RGBA
                  || *p_chroma == VLC_CODEC_BGRA )
            {
                static const ft_drawing_functions DRAW_RGBA =
                    { .extract = RGBFromXRGB,
                      .fill =    FillRGBAPicture,
                      .blend =   BlendGlyphToRGBA };
                func = &DRAW_RGBA;
            }
            else if( *p_chroma == VLC_CODEC_ARGB
                  || *p_chroma == VLC_CODEC_ABGR)
            {
                static const ft_drawing_functions DRAW_ARGB =
                    { .extract = RGBFromXRGB,
                      .fill =    FillARGBPicture,
                      .blend =   BlendGlyphToARGB };
                func = &DRAW_ARGB;
            }
            else
            {
                subpicture_region_Delete(region);
                region = NULL;
                continue;
            }

            RenderAXYZ( p_filter, p_region_in, region, text_block.p_laid,
                                 &renderbbox, &paddedbbox, &bbox, func );
        }

//      if( (bboxcolor & 0xFF) == 0 )
        {
            region->i_x = (paddedbbox.xMin - regionbbox.xMin) + p_region_in->i_x;
            region->i_y = (regionbbox.yMax - paddedbbox.yMax) + p_region_in->i_y;
        }
//      else /* case where the bounding box is larger and visible */
//      {
//          region->i_x = p_region_in->i_x;
//          region->i_y = p_region_in->i_y;
//      }
        region->i_alpha = p_region_in->i_alpha;
        region->i_align = p_region_in->i_align;
        region->b_absolute = p_region_in->b_absolute;
        region->b_in_window = p_region_in->b_in_window;
        break;
    }

    if (region == NULL)
        msg_Warn( p_filter, "no output chroma supported for rendering" );

done:
    FreeLines( text_block.p_laid );

    free( text_block.p_uchars );
    FreeStylesArray( text_block.pp_styles, text_block.i_count );
    if( text_block.pp_ruby )
        FreeRubyBlockArray( text_block.pp_ruby, text_block.i_count );

    return region;
}

static const struct vlc_filter_operations filter_ops =
{
    .render = Render, .close = Destroy,
};

/*****************************************************************************
 * Create: allocates osd-text video thread output method
 *****************************************************************************
 * This function allocates and initializes a Clone vout method.
 *****************************************************************************/
static int Create( filter_t *p_filter )
{
    filter_sys_t  *p_sys            = NULL;

    /* Allocate structure */
    p_filter->p_sys = p_sys = calloc( 1, sizeof( *p_sys ) );
    if( !p_sys )
        return VLC_ENOMEM;

    /* Init Freetype and its stroker */
    if( FT_Init_FreeType( &p_sys->p_library ) )
    {
        msg_Err( p_filter, "Failed to initialize FreeType" );
        free( p_sys );
        return VLC_EGENERIC;
    }

    if( FT_Stroker_New( p_sys->p_library, &p_sys->p_stroker ) )
    {
        msg_Err( p_filter, "Failed to create stroker for outlining" );
        p_sys->p_stroker = NULL;
    }

    p_sys->ftcache = vlc_ftcache_New( VLC_OBJECT(p_filter), p_sys->p_library,
                            var_InheritInteger( p_filter, "freetype-cache-size" ) );
    if( !p_sys->ftcache )
        goto error;

    p_sys->i_scale = 100;

    /* default style to apply to incomplete segments styles */
    p_sys->p_default_style = text_style_Create( STYLE_FULLY_SET );
    if(unlikely(!p_sys->p_default_style))
        goto error;

    /* empty style for style overriding cases */
    p_sys->p_forced_style = text_style_Create( STYLE_NO_DEFAULTS );
    if(unlikely(!p_sys->p_forced_style))
        goto error;

#ifndef HAVE_GET_FONT_BY_FAMILY_NAME
    p_sys->psz_fontfile = var_InheritString( p_filter, "freetype-font" );
    p_sys->psz_monofontfile = var_InheritString( p_filter, "freetype-monofont" );
#endif

    /* fills default and forced style */
    FillDefaultStyles( p_filter );

    if( var_InheritBool( p_filter, "freetype-yuvp" ) )
        p_sys->i_forced_chroma = VLC_CODEC_YUVP;

    /*
     * The following variables should not be cached, as they might be changed on-the-fly:
     * freetype-rel-fontsize, freetype-background-opacity, freetype-background-color,
     * freetype-outline-thickness, freetype-color
     *
     */

    float f_shadow_angle       = var_InheritFloat( p_filter, "freetype-shadow-angle" );
    float f_shadow_distance    = var_InheritFloat( p_filter, "freetype-shadow-distance" );
    f_shadow_distance          = VLC_CLIP( f_shadow_distance, 0, 1 );
    p_sys->f_shadow_vector_x   = f_shadow_distance * cosf((float)(2. * M_PI) * f_shadow_angle / 360);
    p_sys->f_shadow_vector_y   = f_shadow_distance * sinf((float)(2. * M_PI) * f_shadow_angle / 360);

    p_sys->fs = FontSelectNew( p_filter  );
    if( !p_sys->fs )
        goto error;

    if( LoadFontsFromAttachments( p_filter ) == VLC_ENOMEM )
        goto error;

    p_filter->ops = &filter_ops;

    return VLC_SUCCESS;

error:
    Destroy( p_filter );
    return VLC_EGENERIC;
}

/*****************************************************************************
 * Destroy: destroy Clone video thread output method
 *****************************************************************************
 * Clean up all data and library connections
 *****************************************************************************/
static void Destroy( filter_t *p_filter )
{
    filter_sys_t *p_sys = p_filter->p_sys;

#ifdef DEBUG_PLATFORM_FONTS
    if(p_sys->fs)
        DumpFamilies( p_sys->fs );
#endif

    if( p_sys->ftcache )
        vlc_ftcache_Delete( p_sys->ftcache );

    if( p_sys->fs )
        FontSelectDelete( p_sys->fs );

    free( p_sys->psz_fontfile );
    free( p_sys->psz_monofontfile );

    /* Text styles */
    text_style_Delete( p_sys->p_default_style );
    text_style_Delete( p_sys->p_forced_style );

    /* Attachments */
    if( p_sys->pp_font_attachments )
    {
        for( int k = 0; k < p_sys->i_font_attachments; k++ )
            vlc_input_attachment_Release( p_sys->pp_font_attachments[k] );

        free( p_sys->pp_font_attachments );
    }

    /* Freetype */
    if( p_sys->p_stroker )
        FT_Stroker_Done( p_sys->p_stroker );

    FT_Done_FreeType( p_sys->p_library );

    free( p_sys );
}
