// ===== wp_disc.frag =====
#version 450

layout(location = 0) in vec2 qt_TexCoord0;
layout(location = 0) out vec4 fragColor;

layout(binding = 1) uniform sampler2D source1;  // Current wallpaper
layout(binding = 2) uniform sampler2D source2;  // Next wallpaper

layout(std140, binding = 0) uniform buf {
    mat4 qt_Matrix;
    float qt_Opacity;
    float progress;      // Transition progress (0.0 to 1.0)
    float centerX;       // X coordinate of disc center (0.0 to 1.0)
    float centerY;       // Y coordinate of disc center (0.0 to 1.0)
    float smoothness;    // Edge smoothness (0.0 to 1.0, 0=sharp, 1=very smooth)
    float aspectRatio;   // Width / Height of the screen

    // Fill mode parameters
    float fillMode;      // 0=center, 1=crop, 2=fit, 3=stretch, 4=repeat
    float imageWidth1;   // Width of source1 image
    float imageHeight1;  // Height of source1 image
    float imageWidth2;   // Width of source2 image
    float imageHeight2;  // Height of source2 image
    float screenWidth;   // Screen width
    float screenHeight;  // Screen height
    vec4 fillColor;      // Fill color for empty areas (default: black)

    // Solid color mode
    float isSolid1;      // 1.0 if source1 is solid color, 0.0 otherwise
    float isSolid2;      // 1.0 if source2 is solid color, 0.0 otherwise
    vec4 solidColor1;    // Solid color for source1
    vec4 solidColor2;    // Solid color for source2
} ubuf;

// Calculate UV coordinates based on fill mode
vec2 calculateUV(vec2 uv, float imgWidth, float imgHeight) {
    float imageAspect = imgWidth / imgHeight;
    float screenAspect = ubuf.screenWidth / ubuf.screenHeight;
    vec2 transformedUV = uv;

    if (ubuf.fillMode < 0.5) {
        // Mode 0: no (center) - No resize, center image at original size
        // Convert UV to pixel coordinates, offset, then back to UV in image space
        vec2 screenPixel = uv * vec2(ubuf.screenWidth, ubuf.screenHeight);
        vec2 imageOffset = (vec2(ubuf.screenWidth, ubuf.screenHeight) - vec2(imgWidth, imgHeight)) * 0.5;
        vec2 imagePixel = screenPixel - imageOffset;
        transformedUV = imagePixel / vec2(imgWidth, imgHeight);
    }
    else if (ubuf.fillMode < 1.5) {
        // Mode 1: crop (fill/cover) - Fill screen, crop excess (default)
        float scale = max(ubuf.screenWidth / imgWidth, ubuf.screenHeight / imgHeight);
        vec2 scaledImageSize = vec2(imgWidth, imgHeight) * scale;
        vec2 offset = (scaledImageSize - vec2(ubuf.screenWidth, ubuf.screenHeight)) / scaledImageSize;
        transformedUV = uv * (vec2(1.0) - offset) + offset * 0.5;
    }
    else if (ubuf.fillMode < 2.5) {
        // Mode 2: fit (contain) - Fit inside screen, maintain aspect ratio
        float scale = min(ubuf.screenWidth / imgWidth, ubuf.screenHeight / imgHeight);
        vec2 scaledImageSize = vec2(imgWidth, imgHeight) * scale;
        vec2 offset = (vec2(ubuf.screenWidth, ubuf.screenHeight) - scaledImageSize) * 0.5;

        // Convert screen UV to pixel coordinates
        vec2 screenPixel = uv * vec2(ubuf.screenWidth, ubuf.screenHeight);
        // Adjust for offset and scale
        vec2 imagePixel = (screenPixel - offset) / scale;
        // Convert back to UV coordinates in image space
        transformedUV = imagePixel / vec2(imgWidth, imgHeight);
    }
    else if (ubuf.fillMode < 3.5) {
        // Mode 3: stretch - Use original UV (stretches to fit)
        // No transformation needed for stretch mode
    }
    else {
        // Mode 4: repeat (tile) - Tile image at original size
        vec2 screenPixel = uv * vec2(ubuf.screenWidth, ubuf.screenHeight);
        transformedUV = screenPixel / vec2(imgWidth, imgHeight);
    }

    return transformedUV;
}

// Sample texture with fill mode and handle out-of-bounds
// If isSolid > 0.5, returns solidColor instead of sampling texture
vec4 sampleWithFillMode(sampler2D tex, vec2 uv, float imgWidth, float imgHeight, float isSolid, vec4 solidColor) {
    // Return solid color if in solid color mode
    if (isSolid > 0.5) {
        return solidColor;
    }

    vec2 transformedUV = calculateUV(uv, imgWidth, imgHeight);

    // Mode 4 (repeat): use fract() to tile the image
    if (ubuf.fillMode > 3.5) {
        return texture(tex, fract(transformedUV));
    }

    // Check if UV is out of bounds
    if (transformedUV.x < 0.0 || transformedUV.x > 1.0 ||
        transformedUV.y < 0.0 || transformedUV.y > 1.0) {
        return ubuf.fillColor;
    }

    return texture(tex, transformedUV);
}

void main() {
    vec2 uv = qt_TexCoord0;

    // Sample textures with fill mode (handles solid colors)
    vec4 color1 = sampleWithFillMode(source1, uv, ubuf.imageWidth1, ubuf.imageHeight1, ubuf.isSolid1, ubuf.solidColor1);
    vec4 color2 = sampleWithFillMode(source2, uv, ubuf.imageWidth2, ubuf.imageHeight2, ubuf.isSolid2, ubuf.solidColor2);

    // Map smoothness from 0.0-1.0 to 0.001-0.5 range
    // Using a non-linear mapping for better control
    float mappedSmoothness = mix(0.001, 0.5, ubuf.smoothness * ubuf.smoothness);

    // Adjust UV coordinates to compensate for aspect ratio
    // This makes distances circular instead of elliptical
    vec2 adjustedUV = vec2(uv.x * ubuf.aspectRatio, uv.y);
    vec2 adjustedCenter = vec2(ubuf.centerX * ubuf.aspectRatio, ubuf.centerY);
    
    // Calculate distance in aspect-corrected space
    float dist = distance(adjustedUV, adjustedCenter);
    
    // Calculate the maximum possible distance (corner to corner)
    // This ensures the disc can cover the entire screen
    float maxDistX = max(ubuf.centerX * ubuf.aspectRatio, 
                         (1.0 - ubuf.centerX) * ubuf.aspectRatio);
    float maxDistY = max(ubuf.centerY, 1.0 - ubuf.centerY);
    float maxDist = length(vec2(maxDistX, maxDistY));
    
    // Scale progress to cover the maximum distance
    // Add extra range for smoothness to ensure complete coverage
    // Adjust smoothness for aspect ratio to maintain consistent visual appearance
    float adjustedSmoothness = mappedSmoothness * max(1.0, ubuf.aspectRatio);
    
    // Start the radius from -adjustedSmoothness, this ensures the disc is completely hidden when progress=0
    float totalDistance = maxDist + 2.0 * adjustedSmoothness;
    float radius = -adjustedSmoothness + ubuf.progress * totalDistance;

    // Use smoothstep for a smooth edge transition
    float factor = smoothstep(radius - adjustedSmoothness, radius + adjustedSmoothness, dist);
    
    // Mix the textures (factor = 0 inside disc, 1 outside)
    fragColor = mix(color2, color1, factor);
    
    fragColor *= ubuf.qt_Opacity;
}