/*
 * ControlLayout.h - layout for controls
 *
 * Copyright (c) 2019-2019 Johannes Lorenz <j.git$$$lorenz-ho.me, $$$=@>
 *
 * This file is part of LMMS - https://lmms.io
 *
 * This program is free software; you can redistribute it and/or
 * modify it under the terms of the GNU General Public
 * License as published by the Free Software Foundation; either
 * version 2 of the License, or (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 * General Public License for more details.
 *
 * You should have received a copy of the GNU General Public
 * License along with this program (see COPYING); if not, write to the
 * Free Software Foundation, Inc., 51 Franklin Street, Fifth Floor,
 * Boston, MA 02110-1301 USA.
 *
 */

/****************************************************************************
**
** Copyright (C) 2016 The Qt Company Ltd.
** Contact: https://www.qt.io/licensing/
**
** $QT_BEGIN_LICENSE:BSD$
** Commercial License Usage
** Licensees holding valid commercial Qt licenses may use this file in
** accordance with the commercial license agreement provided with the
** Software or, alternatively, in accordance with the terms contained in
** a written agreement between you and The Qt Company. For licensing terms
** and conditions see https://www.qt.io/terms-conditions. For further
** information use the contact form at https://www.qt.io/contact-us.
**
** BSD License Usage
** Alternatively, you may use this file under the terms of the BSD license
** as follows:
**
** "Redistribution and use in source and binary forms, with or without
** modification, are permitted provided that the following conditions are
** met:
**   * Redistributions of source code must retain the above copyright
**     notice, this list of conditions and the following disclaimer.
**   * Redistributions in binary form must reproduce the above copyright
**     notice, this list of conditions and the following disclaimer in
**     the documentation and/or other materials provided with the
**     distribution.
**   * Neither the name of The Qt Company Ltd nor the names of its
**     contributors may be used to endorse or promote products derived
**     from this software without specific prior written permission.
**
**
** THIS SOFTWARE IS PROVIDED BY THE COPYRIGHT HOLDERS AND CONTRIBUTORS
** "AS IS" AND ANY EXPRESS OR IMPLIED WARRANTIES, INCLUDING, BUT NOT
** LIMITED TO, THE IMPLIED WARRANTIES OF MERCHANTABILITY AND FITNESS FOR
** A PARTICULAR PURPOSE ARE DISCLAIMED. IN NO EVENT SHALL THE COPYRIGHT
** OWNER OR CONTRIBUTORS BE LIABLE FOR ANY DIRECT, INDIRECT, INCIDENTAL,
** SPECIAL, EXEMPLARY, OR CONSEQUENTIAL DAMAGES (INCLUDING, BUT NOT
** LIMITED TO, PROCUREMENT OF SUBSTITUTE GOODS OR SERVICES; LOSS OF USE,
** DATA, OR PROFITS; OR BUSINESS INTERRUPTION) HOWEVER CAUSED AND ON ANY
** THEORY OF LIABILITY, WHETHER IN CONTRACT, STRICT LIABILITY, OR TORT
** (INCLUDING NEGLIGENCE OR OTHERWISE) ARISING IN ANY WAY OUT OF THE USE
** OF THIS SOFTWARE, EVEN IF ADVISED OF THE POSSIBILITY OF SUCH DAMAGE."
**
** $QT_END_LICENSE$
**
****************************************************************************/

#ifndef LMMS_GUI_CONTROL_LAYOUT_H
#define LMMS_GUI_CONTROL_LAYOUT_H

#include <QLayout>
#include <QMultiMap>
#include <QStyle>

class QLayoutItem;
class QRect;
class QString;
class QLineEdit;


namespace lmms::gui
{

/**
	Layout for controls (models)

	Originally token from Qt's FlowLayout example. Modified.

	Features a search bar, as well as looking up widgets with string keys
	Keys have to be provided in the widgets' objectNames
*/
class ControlLayout : public QLayout
{
	Q_OBJECT

#if (QT_VERSION >= QT_VERSION_CHECK(6, 0, 0))
	using ControlLayoutMap = QMultiMap<QString, QLayoutItem*>;
	using ControlLayoutMapIterator = QMultiMapIterator<QString, QLayoutItem*>;
#else
	using ControlLayoutMap = QMap<QString, QLayoutItem*>;
	using ControlLayoutMapIterator = QMapIterator<QString, QLayoutItem*>;
#endif

public:
	explicit ControlLayout(QWidget *parent,
		int margin = -1, int hSpacing = -1, int vSpacing = -1);
	~ControlLayout() override;

	void addItem(QLayoutItem *item) override;
	int horizontalSpacing() const;
	int verticalSpacing() const;
	Qt::Orientations expandingDirections() const override;
	bool hasHeightForWidth() const override;
	int heightForWidth(int) const override;
	int count() const override;
	QLayoutItem *itemAt(int index) const override;
	QLayoutItem *itemByString(const QString& key) const;
	QSize minimumSize() const override;
	void setGeometry(const QRect &rect) override;
	QSize sizeHint() const override;
	QLayoutItem *takeAt(int index) override;
	//! remove focus from QLineEdit search bar
	//! this may be useful if the mouse is outside the layout
	void removeFocusFromSearchBar();

private slots:
	void onTextChanged(const QString&);

private:
	int doLayout(const QRect &rect, bool testOnly) const;
	int smartSpacing(QStyle::PixelMetric pm) const;
	ControlLayoutMap::const_iterator pairAt(int index) const;

	ControlLayoutMap m_itemMap;
	int m_hSpace;
	int m_vSpace;
	// relevant dimension is width, as later, heightForWidth() will be called
	// 400 looks good and is ~4 knobs in a row
	constexpr const static int m_minWidth = 400;
	QLineEdit* m_searchBar;
	//! name of search bar, must be ASCII sorted before any alpha numerics
	static constexpr const char* s_searchBarName = "!!searchBar!!";
};

} // namespace lmms::gui

#endif // LMMS_GUI_CONTROL_LAYOUT_H
