//--------------------------------------------------------------
// LumaWipe
//
// Copyright (c) LWKS Software Ltd.  All Rights Reserved
//--------------------------------------------------------------
#include "_utils.fx"

DeclareLightworksEffect( "Luma Wipe", "Mix", "Components", kNoNotes, "CanSize" );

//--------------------------------------------------------------
// Params
//--------------------------------------------------------------
DeclareIntParam( SetTechnique, "Luma source", kNoGroup, 0, "Foreground|Background" );
DeclareFloatParam( BorderWidth, "Size", "Border", kNoFlags, 0.0, 0.0, 1.0 );
DeclareColourParam( BorderColour1, "First colour", "Border", kNoFlags, 1.0, 1.0, 1.0, 1.0 );
DeclareColourParam( BorderColour2, "Second colour", "Border", kNoFlags, 1.0, 1.0, 1.0, 1.0 );
DeclareFloatParam( Softness, "Softness", kNoGroup, kNoFlags, 0.2, 0.0, 1.0 );
DeclareFloatParamAnimated( Progress, "Progress", kNoGroup, kNoFlags, 0.0, 0.0, 1.0 );
DeclareBoolParam( Invert, "Invert", kNoGroup, false );

//--------------------------------------------------------------
// Inputs
//--------------------------------------------------------------
DeclareInputs( fg, bg );
DeclareMask;

//-----------------------------------------------------------------------------------------
float4 doWipe( float4 fgPix, float4 bgPix, float luma )
{
   float totalEdgeSize = Softness + BorderWidth;
   float halfSoftness  = ( Softness / 2.0 );
   float edgeSoftStart = ( Progress * ( 1.0 + totalEdgeSize ) ) - totalEdgeSize;
   float borderStart   = edgeSoftStart + halfSoftness;
   float borderEnd     = borderStart + BorderWidth;
   float edgeSoftEnd   = borderEnd + halfSoftness;

   float4 ret;

   luma = lerp( luma, 1.0 - luma, Invert );

   if ( luma > edgeSoftEnd )
   {
      ret = fgPix;
   }
   else if ( luma < edgeSoftStart )
   {
      ret = bgPix;
   }
   else
   {
      float lumaMinusEdgeStart =  luma - edgeSoftStart;

      if ( BorderWidth == 0 )
      {
         ret = lerp( bgPix, fgPix, lumaMinusEdgeStart / Softness );
      }
      else
      {
         float4 borderColour = lerp( BorderColour1, BorderColour2, lumaMinusEdgeStart / totalEdgeSize );

         if ( luma < borderStart )
            ret = lerp( bgPix, borderColour, lumaMinusEdgeStart / halfSoftness );
         else if ( luma > borderEnd )
            ret = lerp( fgPix, borderColour, ( edgeSoftEnd - luma ) / halfSoftness );
         else
            ret = borderColour;
      }
   }

   return ret;
}


//--------------------------------------------------------------//
// Pixel Shader
//--------------------------------------------------------------//
DeclareEntryPoint( fg_main )
{
   float4 fgPix = ReadPixel( fg, uv1 );
   float4 bgPix = ReadPixel( bg, uv2 );

   float4 result = doWipe( fgPix, bgPix, ( fgPix.r + fgPix.g + fgPix.b ) / 3.0 );
   return lerp( fgPix, result, tex2D( Mask, uv1 ) );
}
//--------------------------------------------------------------//
DeclareEntryPoint( bg_main )
{
   float4 fgPix = ReadPixel( fg, uv1 );
   float4 bgPix = ReadPixel( bg, uv2 );

   float4 result = doWipe( fgPix, bgPix, ( bgPix.r + bgPix.g + bgPix.b ) / 3.0 );
   return lerp( fgPix, result, tex2D( Mask, uv1 ) );
}
