//--------------------------------------------------------------//
// Colour Correction 4
//
// Copyright (c) LWKS Software Ltd.  All Rights Reserved
//--------------------------------------------------------------//
DeclareLightworksEffect( "Colour Correction", "Colour", "Components", kNoNotes, CanSize );

//--------------------------------------------------------------
// Params
//--------------------------------------------------------------
DeclareColourParam( BlackLevel,       "Black Level", kNoGroup, kNoFlags, 0,0,0,1 );
DeclareColourParam( GreyLevel,        "Grey Level",  kNoGroup, kNoFlags, 0.5,0.5,0.5,1 );
DeclareColourParam( WhiteLevel ,      "White Level", kNoGroup, kNoFlags, 1,1,1,1 );
DeclareColourParam( ShadowTintColour, "Black Level", kNoGroup, kNoFlags, 0,0,0,1 );
DeclareColourParam( MidTintColour,    "Black Level", kNoGroup, kNoFlags, 0,0,0,1 );
DeclareColourParam( HighTintColour,   "Black Level", kNoGroup, kNoFlags, 0,0,0,1 );
DeclareFloatParam( Saturation,        "Saturation",  kNoGroup, kNoFlags, 0,0,0 );
DeclareColourParam( Gamma,            "Gamma",       kNoGroup, kNoFlags, 0,0,0,0 );
DeclareColourParam( Contrast,         "Contrast",    kNoGroup, kNoFlags, 0,0,0,0 );
DeclareColourParam( Brightness,       "Brightness",  kNoGroup, kNoFlags, 0,0,0,0 );
DeclareColourParam( Gain,             "Gain",        kNoGroup, kNoFlags, 0,0,0,0 );
DeclareFloatParam( HueBias,           "Hue Bias",    kNoGroup, kNoFlags, 0,0,0 );
DeclareFloatParam( SatBias,           "Sat Bias",    kNoGroup, kNoFlags, 0,0,0 );
DeclareFloatParam( ValBias,           "Val Bias",    kNoGroup, kNoFlags, 0,0,0 );
DeclareFloatParam( SatScale,          "Sat scale",   kNoGroup, kNoFlags, 0,0,0 );
DeclareFloatParam( ValScale,          "Val scale ",  kNoGroup, kNoFlags, 0,0,0 );

float _offset = 0.5 / 256.0;
float _scale  = 255.0 / 256.0;
float4 _k1 = float4( 0.0, -1.0 / 3.0, 2.0 / 3.0, -1.0 );
float4 _k2 = float4( 1.0f, 2.0f / 3.0f, 1.0f / 3.0f, 3.0f );

//--------------------------------------------------------------//
// Inputs
//--------------------------------------------------------------//
DeclareInput( Input );
DeclareMask;
DeclareInput( Lut );

//--------------------------------------------------------------
// CCFull
//--------------------------------------------------------------
float4 CCFull( float4 src )
{
   float4 ret = src;

   // Do balancing
   
   float greyAverage = ( GreyLevel.r + GreyLevel.g + GreyLevel.b ) / 3.0;
   ret -= BlackLevel;
   ret /= ( WhiteLevel - BlackLevel );
   ret -= GreyLevel;
   ret += greyAverage;
   ret = clamp( ret, 0.0, 1.0 );

   // Calc its brightness (so that it can be treated as a shadow/midtone/highlight as appropriate)
   
   float lum = ( ret.r + ret.g + ret.b ) / 3.0;

   // Apply the shadow/midtone/highlight tints
   
   float2 lutPos   = float2( ( lum * _scale ) + _offset, 0.5 );
   float4 lutPixel = tex2D( Lut, lutPos );
   ret += ShadowTintColour * lutPixel.r;
   ret += MidTintColour    * lutPixel.g;
   ret += HighTintColour   * lutPixel.b;

   float avg = ( ret.r + ret.g + ret.b ) / 3.0;
   float4 avg4 = float4( avg, avg, avg, avg );
   ret = avg4 + ( ( ret - avg4 ) * Saturation );

   // Do brightness, gamma & contrast logic
   
   ret = ( ( ( ( pow( saturate( ret ), 1 / Gamma ) * Gain ) + Brightness ) - 0.5 ) * Contrast ) + 0.5;

   // Convert to HSV
   //
   // See https://gamedev.stackexchange.com/questions/59797/glsl-shader-change-hue-saturation-brightness

   float4 p = lerp( float4( ret.bg, _k1.wz ), float4( ret.gb, _k1.xy ), step( ret.b, ret.g ) );
   float4 q = lerp( float4( p.xyw, ret.r ),   float4( ret.r, p.yzx ),   step( p.x,   ret.r ) );

   float d = q.x - min( q.w, q.y );
   float e = 1.0e-10;

   float3 hsv = float3( abs( q.z + ( q.w - q.y ) / ( 6.0f * d + e ) ), d / ( q.x + e ), q.x );

   // Scale and bias 

   hsv.x = frac( hsv.x + HueBias );
   hsv.y = saturate( hsv.y * SatScale + SatBias );
   hsv.z = saturate( hsv.z * ValScale + ValBias );

   // Convert back to RGB

   float3 p3 = abs( frac( hsv.xxx + _k2.xyz ) * 6.0f - _k2.www );
                                                        
   ret.rgb = hsv.z * lerp( _k2.xxx, saturate( p3 - _k2.xxx ), hsv.y );

   return ret;
}

//--------------------------------------------------------------
// Entry points
//--------------------------------------------------------------
DeclareEntryPoint( CC )
{
   // Read the source image pixel
   float4 src = ReadPixel( Input, uv1 );
   float4 ret = src;

   // Do balancing
   float greyAverage = ( GreyLevel.r + GreyLevel.g + GreyLevel.b ) / 3.0;
   ret -= BlackLevel;
   ret /= ( WhiteLevel - BlackLevel );
   ret -= GreyLevel;
   ret += greyAverage;
   ret = clamp( ret, 0.0, 1.0 );

   // Calc its brightness (so that it can be treated as a shadow/midtone/highlight as appropriate)
   float lum = ( ret.r + ret.g + ret.b ) / 3.0;

   // Apply the shadow/midtone/highlight tints
   float2 lutPos   = float2( ( lum * _scale ) + _offset, 0.5 );
   float4 lutPixel = tex2D( Lut, lutPos );
   ret += ShadowTintColour * lutPixel.r;
   ret += MidTintColour    * lutPixel.g;
   ret += HighTintColour   * lutPixel.b;

   float avg = ( ret.r + ret.g + ret.b ) / 3.0;
   float4 avg4 = float4( avg, avg, avg, avg );
   ret = avg4 + ( ( ret - avg4 ) * Saturation );

   // Do brightness, gamma & contrast logic
   ret = ( ( ( ( pow( saturate( ret ), 1 / Gamma ) * Gain ) + Brightness ) - 0.5 ) * Contrast ) + 0.5;

   ret.a = src.a;

   return ret;
}
//--------------------------------------------------------------
DeclareEntryPoint( CCMasked )
{
   // Read the source image pixel
   float4 src = ReadPixel( Input, uv1 );
   float4 ret = src;

   // Do balancing
   float greyAverage = ( GreyLevel.r + GreyLevel.g + GreyLevel.b ) / 3.0;
   ret -= BlackLevel;
   ret /= ( WhiteLevel - BlackLevel );
   ret -= GreyLevel;
   ret += greyAverage;
   ret = clamp( ret, 0.0, 1.0 );

   // Calc its brightness (so that it can be treated as a shadow/midtone/highlight as appropriate)
   float lum = ( ret.r + ret.g + ret.b ) / 3.0;

   // Apply the shadow/midtone/highlight tints
   float2 lutPos   = float2( ( lum * _scale ) + _offset, 0.5 );
   float4 lutPixel = tex2D( Lut, lutPos );
   ret += ShadowTintColour * lutPixel.r;
   ret += MidTintColour    * lutPixel.g;
   ret += HighTintColour   * lutPixel.b;

   float avg = ( ret.r + ret.g + ret.b ) / 3.0;
   float4 avg4 = float4( avg, avg, avg, avg );
   ret = avg4 + ( ( ret - avg4 ) * Saturation );

   // Do brightness, gamma & contrast logic
   ret = ( ( ( ( pow( saturate( ret ), 1 / Gamma ) * Gain ) + Brightness ) - 0.5 ) * Contrast ) + 0.5;

   ret.a = src.a;

   return lerp( src, ret, tex2D( Mask, uv2 ) );
}
//--------------------------------------------------------------
DeclareEntryPoint( H2R2H )
{
   return CCFull( ReadPixel( Input, uv1 ) );
}
//--------------------------------------------------------------
DeclareEntryPoint( H2R2HMasked )
{
   float4 src = ReadPixel( Input, uv1 );
   float4 ret = CCFull( src );

   return lerp( src, ret, tex2D( Mask, uv2 ) );
}
